/*
Collision Mask Utilities
Copyright ©2014 David Powell

Released under the MIT Licence

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "mask.h"

const quint8 Mask::signature[10] = { 139, 67, 77, 83, 75, 13, 10, 26, 10, 0 };

/*!
 * \brief A Mask object
 */
Mask::Mask() :
    width(0),
    height(0)
{ }

/*!
 * \brief Load the specified mask file
 * \param filename The mask file to load
 */
void Mask::load(QString filename)
{
    QFile file(filename);

    if (file.open(QIODevice::ReadOnly) == false)
    {
        throw std::runtime_error("The mask file could not be loaded.");
    }

    QByteArray maskData = file.readAll();

    if (checkSignature(maskData) == false)
    {
        throw std::runtime_error("The file is not a mask, or the file is corrupt.");
    }

    quint32 version = qFromBigEndian<quint32>((const uchar *)maskData.mid(VersionOffset, 4).constData());
    if (version != 1)
    {
        throw std::runtime_error("The mask file cannot be used as only version 1 CMSK files are supported.");
    }

    quint32 crc = qFromBigEndian<quint32>((const uchar *)maskData.right(4).constData());
    quint32 calculatedCrc = calculateCrc32(maskData.left(maskData.length() - 4));

    if (crc != calculatedCrc)
    {
        throw std::runtime_error("The mask file is corrupt, the CRC check failed.");
    }

    quint32 dataOffset = qFromBigEndian<quint32>((const uchar *)maskData.mid(DataOffsetOffset, 4).constData());
    quint32 dataLength = qFromBigEndian<quint32>((const uchar *)maskData.mid(DataLengthOffset, 4).constData());
    this->width = qFromBigEndian<quint32>((const uchar *)maskData.mid(WidthOffset, 4).constData());
    this->height = qFromBigEndian<quint32>((const uchar *)maskData.mid(HeightOffset, 4).constData());

    this->data = rleDecode(maskData, dataOffset, dataLength, this->width * this->height);
}

/*!
 * \brief Check the signature of the supplied mask data
 * \param maskData The mask data to check
 * \return true if the signature is correct, false otherwise
 */
bool Mask::checkSignature(QByteArray maskData)
{
    for (int offset = 0; offset < 10; offset++)
    {
        if (signature[offset] != (quint8)maskData[offset])
            return false;
    }

    return true;
}

/*!
 * \brief Decode the RLE data contained within the mask data
 * \param maskData The mask data that contains the RLE data to decode
 * \param offset The start of the RLE data within the mask data
 * \param length The length of the RLE data
 * \param decodedLength The length of the decoded RLE data
 * \return The decoded data
 */
QByteArray Mask::rleDecode(QByteArray maskData, quint32 offset, quint32 length, quint32 decodedLength)
{
    QByteArray rleData = maskData.mid(offset, length);
    QDataStream inStream(&rleData, QIODevice::ReadOnly);

    QByteArray decodedData(decodedLength, 0);
    QDataStream outStream(&decodedData, QIODevice::WriteOnly);

    quint8 lastPixel;
    inStream >> lastPixel;

    while (inStream.atEnd() == false)
    {
        quint8 pixel;
        inStream >> pixel;

        if (pixel != lastPixel)
        {
            outStream << lastPixel;
            lastPixel = pixel;
        }
        else
        {
            quint8 count;
            inStream >> count;

            for (quint8 num = 0; num < count; num++)
            {
                outStream << pixel;
            }

            if (inStream.atEnd() == false)
            {
                inStream >> lastPixel;
            }
        }
    }

    if (outStream.atEnd() == false)
    {
        outStream << lastPixel;
    }

    if (outStream.atEnd() == false)
    {
        throw std::runtime_error("The mask file is corrupt.");
    }

    return decodedData;
}

/*!
 * \brief Calculate the CRC-32 checksum for the supplied data
 * \param data The data to checksum
 * \param previousCrc32 The previous CRC-32 checksum,
 *        if calculating a checksum in chunks
 * \return The CRC-32 checksum
 */
quint32 Mask::calculateCrc32(QByteArray data, quint32 previousCrc32)
{
    const quint32 Polynomial = 0xEDB88320;
    int index = 0;
    quint32 crc = ~previousCrc32;

    while (index < data.count())
    {
        crc ^= (quint8)data[index++];
        for (quint32 bit = 0; bit < 8; bit++)
        {
            crc = (crc >> 1) ^ (-qint32(crc & 1) & Polynomial);
        }
    }

    return ~crc;
}
